program Attribute4;

{$APPTYPE CONSOLE}

uses
  System.SysUtils,
  System.Rtti,
  System.TypInfo,
  System.Generics.Collections,
  System.RegularExpressions;

type
  // Base per tutti i validator-attribute
  TValidatorAttribute = class(TCustomAttribute)
  public
    // aValue pu essere Empty/Null/Unassigned: gestiscilo nel derivato
    function Validate(const aName: string; const aValue: TValue; out aError: string): Boolean; virtual; abstract;
  end;

  // Validatore email con opzione Required
  TEmailAttribute = class(TValidatorAttribute)
  private
    FRequired: Boolean;
    FPattern: string;
  public
    constructor Create(const aRequired: Boolean = True; const aPattern: string = '');
    function Validate(const aName: string; const aValue: TValue; out aError: string): Boolean; override;
    property Required: Boolean read FRequired;
    property Pattern: string read FPattern;
  end;

constructor TEmailAttribute.Create(const aRequired: Boolean; const aPattern: string);
begin
  FRequired := aRequired;
  // Regex semplice e robusta per uso applicativo (non perfetta RFC5322, ma pratica).
  // Puoi sostituirla passando aPattern nel costruttore.
  if aPattern <> '' then
    FPattern := aPattern
  else
    FPattern := '^[A-Z0-9._%+\-]+@[A-Z0-9.\-]+\.[A-Z]{2,}$';
end;

function TEmailAttribute.Validate(const aName: string; const aValue: TValue; out aError: string): Boolean;
var
  S: string;
begin
  aError := '';
  // Se non richiesto e vuoto -> ok
  if (not FRequired) and (aValue.IsEmpty or (aValue.Kind = tkUString) and (aValue.AsString.Trim = '')) then
    Exit(True);

  // Se richiesto ma vuoto -> errore
  if aValue.IsEmpty or ((aValue.Kind = tkUString) and (aValue.AsString.Trim = '')) then
  begin
    aError := Format('%s  obbligatorio.', [aName]);
    Exit(False);
  end;

  // Tipo atteso: string
  if aValue.Kind <> tkUString then
  begin
    aError := Format('%s non  una stringa.', [aName]);
    Exit(False);
  end;

  S := aValue.AsString.Trim;
  // Confronto case-insensitive
  if not TRegEx.IsMatch(S, FPattern, [roIgnoreCase]) then
  begin
    aError := Format('%s non  un indirizzo email valido.', [aName]);
    Exit(False);
  end;

  Result := True;
end;

// --- Modello di esempio

type
  TUser = class
  private
    FEmail: string;
  public
    [TEmailAttribute(True)] // required
    property Email: string read FEmail write FEmail;

    // altri campi/propriet...
  end;

// --- Validazione via RTTI su propriet e campi decorati

function GetDisplayName(const aMember: TRttiMember): string;
begin
  // Nome leggibile per messaggi di errore (puoi mappare con un attributo DisplayName personalizzato)
  Result := aMember.Name;
end;

procedure ValidateObject(const aInstance: TObject; const aErrors: TList<string>);
var
  lCtx: TRttiContext;
  lType: TRttiType;
  lProp: TRttiProperty;
  lField: TRttiField;
  lAttr: TCustomAttribute;
  lValue: TValue;
  lErr: string;
begin
  if (aInstance = nil) or (aErrors = nil) then
    Exit;

  lCtx := TRttiContext.Create;
  lType := lCtx.GetType(aInstance.ClassType);

  // Propriet
  for lProp in lType.GetProperties do
  begin
    // Consideriamo solo propriet leggibili (getter disponibile)
    if not lProp.IsReadable then
      Continue;

    for lAttr in lProp.GetAttributes do
      if lAttr is TValidatorAttribute then
      begin
        lValue := lProp.GetValue(aInstance);
        if not TValidatorAttribute(lAttr).Validate(GetDisplayName(lProp), lValue, lErr) then
          aErrors.Add(lErr);
      end;
  end;

  // Campi (se vuoi supportarli)
  for lField in lType.GetFields do
  begin
    for lAttr in lField.GetAttributes do
      if lAttr is TValidatorAttribute then
      begin
        lValue := lField.GetValue(aInstance);
        if not TValidatorAttribute(lAttr).Validate(GetDisplayName(lField), lValue, lErr) then
          aErrors.Add(lErr);
      end;
  end;
end;

// --- Demo

procedure Demo(const aEmailToTest: string);
var
  lUser: TUser;
  lErrors: TList<string>;
  lMsg: string;
  lI: Integer;
begin
  lUser := TUser.Create;
  try
    lUser.Email := aEmailToTest;

    lErrors := TList<string>.Create;
    try
      ValidateObject(lUser, lErrors);

      if lErrors.Count = 0 then
        Writeln(Format('OK: %s  valido.', [lUser.Email]))
      else
      begin
        Writeln('Errori:');
        for lI := 0 to lErrors.Count - 1 do
        begin
          lMsg := lErrors[lI];
          Writeln('- ' + lMsg);
        end;
      end;
    finally
      lErrors.Free;
    end;
  finally
    lUser.Free;
  end;
end;

begin
  try
    // Prove
    Demo('');                       // richiesto -> errore
    Demo('not-an-email');           // formato -> errore
    Demo('user@example.com');       // ok

    Writeln('--- Fine ---');
    Readln;
  except
    on E: Exception do
    begin
      Writeln('Errore: ' + E.ClassName + ' - ' + E.Message);
      Readln;
    end;
  end;
end.
