unit Utility;

interface

uses
  System.SysUtils,
  System.Rtti,
  FireDAC.Comp.Client,
  Attributes,
  BusinessObjects,
  Vcl.StdCtrls,
  System.Classes,
  System.UITypes,
  Vcl.Graphics,
  Vcl.NumberBox;
type

  TUtility = class
  public
    class procedure SaveToDatabase(aObject: TObject; aQuery: TFDQuery);
    class procedure LoadFromDatabase(aObject: TObject; aQuery: TFDQuery; aId: Integer);
    class procedure BindControlsToObject(aObject: TObject; aComponent: TComponent; const aPrefix: String);
    class function MandatoryOK(aComponent: TComponent): Boolean;
  end;

implementation

{ TORMHelper }

class procedure TUtility.SaveToDatabase(aObject: TObject; aQuery: TFDQuery);
var
  lCtx: TRttiContext;
  lRttiType: TRttiType;
  lProp: TRttiProperty;
  lTableAttr: TableAttribute;
  lColAttr: ColumnAttribute;
  lTableName: string;
  lSQL, lFields, lValues: string;
  lIsUpdate: Boolean;
  lPKField: string;
  lPKValue: Integer;
  lAttr: TCustomAttribute;
  lFieldArray: TArray<string>;
  lValueArray: TArray<string>;
  i: Integer;
begin
  lCtx := TRttiContext.Create;
  try
    lRttiType := lCtx.GetType(aObject.ClassType);

    // Ottieni il nome della tabella
    lTableName := '';
    for lAttr in lRttiType.GetAttributes do
    begin
      if lAttr is TableAttribute then
      begin
        lTableName := TableAttribute(lAttr).TableName;
        Break;
      end;
    end;

    if lTableName = '' then
      raise Exception.Create('TableAttribute non trovato sulla classe');

    // Costruisci la query SQL
    lFields := '';
    lValues := '';
    lPKField := '';
    lPKValue := 0;
    lIsUpdate := False;

    for lProp in lRttiType.GetProperties do
    begin
      for lAttr in lProp.GetAttributes do
      begin
        if lAttr is ColumnAttribute then
        begin
          lColAttr := ColumnAttribute(lAttr);
          if lColAttr.IsPrimaryKey then
          begin
            lPKField := lColAttr.ColumnName;
            lPKValue := lProp.GetValue(aObject).AsInteger;
            lIsUpdate := lPKValue > 0;
          end
          else
          begin
            if lFields <> '' then
            begin
              lFields := lFields + ', ';
              lValues := lValues + ', ';
            end;

            lFields := lFields + lColAttr.ColumnName;
            var lInvariant := TFormatSettings.Invariant;

            case lProp.PropertyType.TypeKind of
              tkInteger, tkInt64:
                lValues := lValues + IntToStr(lProp.GetValue(aObject).AsInteger);
              tkString, tkUString, tkWString, tkLString:
                lValues := lValues + QuotedStr(lProp.GetValue(aObject).AsString);
              tkFloat:
                lValues := lValues + FloatToStr(lProp.GetValue(aObject).AsExtended, lInvariant);
            end;
          end;
        end;
      end;
    end;

    // Esegui INSERT o UPDATE
    aQuery.Close;

    if lIsUpdate then
    begin
      // UPDATE
      lSQL := 'UPDATE ' + lTableName + ' SET ';
      lFieldArray := lFields.Split([',']);
      lValueArray := lValues.Split([',']);

      for i := 0 to High(lFieldArray) do
      begin
        if i > 0 then
          lSQL := lSQL + ', ';
        lSQL := lSQL + Trim(lFieldArray[i]) + ' = ' + Trim(lValueArray[i]);
      end;

      lSQL := lSQL + ' WHERE ' + lPKField + ' = ' + IntToStr(lPKValue);
    end
    else
    begin
      // INSERT
      lSQL := Format('INSERT INTO %s (%s) VALUES (%s)', [lTableName, lFields, lValues]);
    end;

    aQuery.SQL.Text := lSQL;
    aQuery.ExecSQL;

  finally
    lCtx.Free;
  end;
end;


class procedure TUtility.BindControlsToObject(aObject: TObject; aComponent: TComponent; const aPrefix: String);
var
  lCtx: TRttiContext;
  lRttiType: TRttiType;
  lProp: TRttiProperty;
  lAttr: TCustomAttribute;
  lColAttr: ColumnAttribute;
  lStrLenAttr: StringLengthAttribute;
  lControl: TComponent;
  lEdit: TEdit;
  lPropertyName: string;
  lMandatory: Boolean;
  i: Integer;
begin
  lCtx := TRttiContext.Create;
  try
    lRttiType := lCtx.GetType(aObject.ClassType);

    // Itera su tutte le propriet dell'oggetto
    for lProp in lRttiType.GetProperties do
    begin
      lPropertyName := '';
      lMandatory := false;
      lStrLenAttr := nil;

      // Cerca gli attributi della propriet
      for lAttr in lProp.GetAttributes do
      begin
        lMandatory := lAttr is MandatoryAttribute;
        if lAttr is ColumnAttribute then
          lPropertyName := lProp.Name;

        if lAttr is StringLengthAttribute then
          lStrLenAttr := StringLengthAttribute(lAttr);
      end;

      // Se abbiamo trovato sia il nome della propriet che l'attributo StringLength
      if (lPropertyName <> '') and (lStrLenAttr <> nil) then
      begin
        // Cerca il controllo corrispondente nel form
        // Convenzione: ed + NomePropriet (es: edName per la propriet Name)
        lControl := aComponent.FindComponent(aPrefix + lPropertyName);

        if Assigned(lControl) then
        begin
          // Gestione TEdit
          if lControl is TEdit then
          begin
            lEdit := TEdit(lControl);
            lEdit.Tag := 0;
            lEdit.MaxLength := lStrLenAttr.MaxLength;

            if lMandatory then
            begin
              lEdit.Color := clYellow;
              lEdit.Font.Color := clRed;
              lEdit.Tag := 10;
            end;
          end
          // Gestione TNumberBox
          else if lControl is TNumberBox then
          begin
            TNumberBox(lControl).Tag := 0;

            if lMandatory then
            begin
              TNumberBox(lControl).Color := clYellow;
              TNumberBox(lControl).Font.Color := clRed;
              TNumberBox(lControl).Tag := 10;
            end;
          end;
        end;
      end;
    end;

  finally
    lCtx.Free;
  end;
end;

class procedure TUtility.LoadFromDatabase(aObject: TObject; aQuery: TFDQuery; aId: Integer);
var
  lCtx: TRttiContext;
  lRttiType: TRttiType;
  lProp: TRttiProperty;
  lTableAttr: TableAttribute;
  lColAttr: ColumnAttribute;
  lTableName, lPKField, lSQL: string;
  lAttr: TCustomAttribute;
begin
  lCtx := TRttiContext.Create;
  try
    lRttiType := lCtx.GetType(aObject.ClassType);

    // Ottieni il nome della tabella
    for lAttr in lRttiType.GetAttributes do
      if lAttr is TableAttribute then
        lTableName := TableAttribute(lAttr).TableName;

    // Trova il campo chiave primaria
    for lProp in lRttiType.GetProperties do
      for lAttr in lProp.GetAttributes do
        if (lAttr is ColumnAttribute) and ColumnAttribute(lAttr).IsPrimaryKey then
          lPKField := ColumnAttribute(lAttr).ColumnName;

    // Esegui la query
    lSQL := Format('SELECT * FROM %s WHERE %s = %d', [lTableName, lPKField, aId]);
    aQuery.Close;
    aQuery.Open(lSQL);

    if not aQuery.Eof then
    begin
      // Carica i dati nell'oggetto
      for lProp in lRttiType.GetProperties do
      begin
        for lAttr in lProp.GetAttributes do
        begin
          if lAttr is ColumnAttribute then
          begin
            lColAttr := ColumnAttribute(lAttr);

            case lProp.PropertyType.TypeKind of
              tkInteger, tkInt64:
                lProp.SetValue(aObject, aQuery.FieldByName(lColAttr.ColumnName).AsInteger);
              tkString, tkUString, tkWString, tkLString:
                lProp.SetValue(aObject, aQuery.FieldByName(lColAttr.ColumnName).AsString);
              tkFloat:
              begin
                if lProp.PropertyType.Handle = TypeInfo(Currency) then
                  //  Currency
                  lProp.SetValue(aObject, aQuery.FieldByName(lColAttr.ColumnName).AsCurrency)
                else
                  // Altri tipi float
                  lProp.SetValue(aObject, aQuery.FieldByName(lColAttr.ColumnName).AsFloat);
              end;
            end;
          end;
        end;
      end;
    end;

  finally
    lCtx.Free;
  end;
end;

class function TUtility.MandatoryOK(aComponent: TComponent): Boolean;
begin
  Result := true;
  for var I := 0 to aComponent.ComponentCount - 1 do
    if (aComponent.Components[I] is TEdit) and  (aComponent.Components[I].Tag = 10) then
       Result := not Trim(TEdit(aComponent.Components[I]).Text).IsEmpty
end;

end.
