unit Server.AiEngines.Tools;

interface

uses
  System.Classes, System.SysUtils, System.Rtti, System.Generics.Collections,

  Neon.Core.Utils,
  Neon.Core.Attributes,
  Neon.Core.Persistence,
  Neon.Core.Persistence.JSON,

  Server.AiEngines.Core;

type
  TChatProperty = class(TObject)
  private
    FDescription: string;
    FType: string;
  public
    property &Type: string read FType write FType;
    property Description: string read FDescription write FDescription;
  end;

  TChatProperties = class(TObjectDictionary<string, TChatProperty>)
  end;

  TChatParameters = class(TObject)
  private
    FType: string;
    FProperties: TChatProperties;
    FRequired: TArray<string>;
  public
    constructor Create;
    destructor Destroy; override;

    property &Type: string read FType write FType;
    property Properties: TChatProperties read FProperties write FProperties;
    property Required: TArray<string> read FRequired write FRequired;
  end;

  TChatFunction = class(TObject)
  private
    FName: string;
    FDescription: string;
    FParameters: TChatParameters;
  public
    constructor Create;
    destructor Destroy; override;

    property Name: string read FName write FName;
    property Description: string read FDescription write FDescription;
    property Parameters: TChatParameters read FParameters write FParameters;
  end;

  TChatTool = class(TObject)
  private
    FType: string;
    FFunction: TChatFunction;
  public
    constructor Create;
    destructor Destroy; override;

    property &Type: string read FType write FType;
    property &Function: TChatFunction read FFunction write FFunction;
  end;

  TChatTools = class(TObjectList<TChatTool>);

  TTools = class(TObject)
  public
    class function CreateToolsSchema(AToolObject: TObject): TChatTools; static;
  end;

implementation

{ TTools }

class function TTools.CreateToolsSchema(AToolObject: TObject): TChatTools;
var
  LRttiContext: TRttiContext;
  LRttiType: TRttiType;
begin
  Result := TChatTools.Create;
  try
    LRttiContext := TRttiContext.Create;
    try
      LRttiType := LRttiContext.GetType(AToolObject.ClassType);
      for var LRttiMethod in LRttiType.GetMethods do
      begin
        var LAttr := TRttiUtils.FindAttribute<ChatToolAttribute>(LRttiMethod);
        if Assigned(LAttr) then
        begin
          var LRequired: TArray<string> := [];
          var LTool := TChatTool.Create;
          Result.Add(LTool);
          LTool.&Type := 'function';
          LTool.&Function.Name := LRttiMethod.Name;
          LTool.&Function.Description := LAttr.Description;
          for var LPram in LRttiMethod.GetParameters do
          begin
            var LParamAttr := TRttiUtils.FindAttribute<ChatToolAttribute>(LPram);
            begin
              var LProperty := TChatProperty.Create;
              LProperty.&Type := 'string';
              if Assigned(LParamAttr) then
                LProperty.Description := LParamAttr.Description
              else
                LProperty.Description := '';
              LTool.&Function.Parameters.Properties.Add(LPram.Name, LProperty);
              LRequired := LRequired + [LPram.Name];
            end;
          end;
          if LTool.&Function.Parameters.Properties.Count > 0 then
          begin
            LTool.&Function.Parameters.Required := LRequired;
            LTool.&Function.Parameters.&Type := 'object';
          end;
        end;
      end;
    finally
      LRttiContext.Free;
    end;
  except
    Result.Free;
    raise;
  end;
end;

{ TChatTool }

constructor TChatTool.Create;
begin
  FFunction := TChatFunction.Create;
end;

destructor TChatTool.Destroy;
begin
  FFunction.Free;
  inherited;
end;

{ TChatFunction }

constructor TChatFunction.Create;
begin
  FParameters := TChatParameters.Create();
end;

destructor TChatFunction.Destroy;
begin
  FParameters.Free;
  inherited;
end;

{ TChatParameters }

constructor TChatParameters.Create;
begin
  FProperties := TChatProperties.Create([doOwnsValues]);
end;

destructor TChatParameters.Destroy;
begin
  FProperties.Free;
  inherited;
end;

end.
