unit Server.AiEngines.OpenAI;

interface

uses
  System.Classes, System.SysUtils, System.Net.URLClient, System.Rtti, System.JSON,

  OpenAI,
  OpenAI.Chat,
  OpenAI.Chat.Functions,
  OpenAI.Models,
  OpenAI.Embeddings,

  Neon.Core.Persistence,
  Neon.Core.Persistence.JSON,

  Dynamo.Core.Rtti,

  Server.AiEngines.Tools,
  Server.AiEngines.Core;

type
  TToolResponse = record
    ToolId: string;
    Content: string;

    constructor Create(const AToolId, AContent: string);
  end;

  [Alias('openai')]
  TOpenAi = class(TInterfacedObject, IAIEngine, IEmbedder)
  private
    FOpenAI: IOpenAI;
    FModelId: string;
    FEmbeddingModelId: string;
    FTools: TArray<TChatToolParam>;
    FToolResponses: TArray<TToolResponse>;
    FToolCalls: TArray<TChatToolCallBuild>;
    function ConvertMessages(const AMessages: TArray<IChatMessage>): TArray<TChatMessageBuild>;
    function ContentToResponse(const AContent, AModelId: string;
      ADone: Boolean): TChatResponse;
    procedure BuildToolsFromObject(AToolObject: TObject);
    procedure HandleToolsCall(AToolObject: TObject; AToolsCall: TArray<OpenAi.Chat.TChatToolCall>);
    procedure SendResponse(const AContent: string; ADone: Boolean; AProc: TChatMessageEvent);
    procedure OpenAiSyncChat(const Value: TArray<TChatMessageBuild>; AToolObject: TObject; AProc: TChatMessageEvent);
    function GetProxySettings: TProxySettings;
    procedure SetProxySettings(const AProxySettings: TProxySettings);
  public
    function GetApiKey: string;
    procedure SetApiKey(const AValue: string);
    procedure Chat(const AMessages: TArray<IChatMessage>; AProc: TChatMessageEvent); overload;
    procedure Chat(const AMessages: TArray<IChatMessage>; AToolObject: TObject; AProc: TChatMessageEvent); overload;
    function GetProvider: string;
    function GetModels: TArray<string>;
    function GetSummary(const AContent: string): string;
    function GetModel: string;

    function GetEmbeddingModel: string;
    function GetEmbeddingFromString(const AValue: string): TArray<Extended>;
    function GetEmbeddingFromStrings(const AValue: TArray<string>): TArray<TArray<Extended>>;

    constructor Create;
    destructor Destroy; override;
  end;

  TChatFunction = class(TInterfacedObject, IChatFunction)
  private
    FToolObject: TObject;
    FChatTool: TChatTool;
    function GetDescription: string;
    function GetName: string;
    function GetParameters: string;
    function GetStrict: Boolean;
  public
    constructor Create(AToolObject: TObject; AChatTool: TChatTool);

    function Execute(const Args: string): string;
  end;

implementation

{ TOpenAi }

uses
  Dynamo.Core.ServiceLocator;

const
  LLM_MODEL_ID = 'gpt-4.1';
  EMBEDDING_MODEL_ID = 'text-embedding-3-small';

procedure TOpenAi.HandleToolsCall(AToolObject: TObject; AToolsCall: TArray<OpenAi.Chat.TChatToolCall>);

  function GetIndexOfParam(AMethodRtti: TRttiMethod; const AParamName: string): Integer;
  begin
    var Index := 0;
    for var LParam in AMethodRtti.GetParameters do
    begin
      if SameText(LParam.Name, AParamName) then
        Exit(Index);
      Inc(Index);
    end;
    raise Exception.CreateFmt('Parameter "%s" not found', [AParamName]);
  end;

begin
  FToolResponses := [];
  FToolCalls := [];
  for var LToolCall in AToolsCall do
  begin
    var LToolRtti := TRttiUtils.Context.GetType(AToolObject.ClassType);
    var LMethodRtti := LToolRtti.GetMethod(LToolCall.&Function.Name);
    var LArgs: TArray<TValue>;
    SetLength(LArgs, Length(LMethodRtti.GetParameters));
    var JsonArguments := TJSONValue.ParseJSONValue(LToolCall.&Function.Arguments, False, True) as TJSONObject;
    try
      for var LArgPair in JsonArguments do
      begin
        LArgs[GetIndexOfParam(LMethodRtti, LArgPair.JsonString.Value)] := LArgPair.JsonValue.Value;
      end;
    finally
      JsonArguments.Free;
    end;

    var LResult := LMethodRtti.Invoke(AToolObject, LArgs);
    FToolResponses := FToolResponses + [TToolResponse.Create(LToolCall.Id, LResult.ToString)];

    var ResponseToolCall := TChatToolCallBuild.Create(LToolCall.Id, 'function', TFunctionCallBuild.Create(LToolCall.&Function.Name, LToolCall.&Function.Arguments));
    FToolCalls := FToolCalls + [ResponseToolCall];

  end;
end;

procedure TOpenAi.SendResponse(const AContent: string; ADone: Boolean; AProc: TChatMessageEvent);
var
  LResponse: TChatResponse;
begin
  LResponse := ContentToResponse(AContent, FModelId, ADone);
  try
    AProc(LResponse);
  finally
    LResponse.Free;
  end;
end;

procedure TOpenAi.OpenAiSyncChat(const Value: TArray<TChatMessageBuild>; AToolObject: TObject; AProc: TChatMessageEvent);
begin
  var Chat := FOpenAI.Chat.Create(
    procedure(Params: TChatParams)
    begin
      Params.Model(FModelId);
      Params.Messages(Value);
      Params.MaxTokens(4096);
      if Assigned(AToolObject) then
      begin
        BuildToolsFromObject(AToolObject);
        Params.Tools(FTools);
      end;
    end
  );
  try
    if Length(Chat.Choices[0].Message.ToolCalls) = 0 then
    begin
      SendResponse(Chat.Choices[0].Message.Content, True, AProc);
    end;
    HandleToolsCall(AToolObject, Chat.Choices[0].Message.ToolCalls)
  finally
    Chat.Free;
  end;

  // If there are some tool response prepare another chat call
  if Length(FToolResponses) > 0 then
  begin
    var LMesssages := Value;
    var LAssistantMessage := TChatMessageBuild.Assistant('');
    LAssistantMessage.ToolCalls := FToolCalls;
    LMesssages := LMesssages + [LAssistantMessage];
    for var LResponse in FToolResponses do
    begin
      var LToolResponse := TChatMessageBuild.Tool(LResponse.Content, LResponse.ToolId);
      LMesssages := LMesssages + [LToolResponse];
    end;

    OpenAiSyncChat(LMesssages, AToolObject, AProc);
  end;

end;

procedure TOpenAi.Chat(const AMessages: TArray<IChatMessage>;
  AToolObject: TObject; AProc: TChatMessageEvent);
begin
  FTools := [];
  FToolResponses := [];

  // At the moment in case of Tools disable Streaming
  if Assigned(AToolObject) then
  begin
    OpenAiSyncChat(ConvertMessages(AMessages), AToolObject, AProc);
  end
  else
  begin
    FOpenAI.Chat.CreateStream(
      procedure(Params: TChatParams)
      begin
        Params.Model(FModelId);
        Params.Messages(ConvertMessages(AMessages));
        Params.MaxTokens(4096);
        if Assigned(AToolObject) then
        begin
          BuildToolsFromObject(AToolObject);
          Params.Tools(FTools);
        end;
        Params.Stream;
      end,
      procedure(var Chat: TChat; IsDone: Boolean; var Cancel: Boolean)
      begin
        if (not IsDone) and Assigned(Chat) then
        begin
          if Chat.Choices[0].Delta.Content <> '' then
            SendResponse(Chat.Choices[0].Delta.Content, IsDone, AProc);
        end
        else if IsDone then
          SendResponse('', IsDone, AProc);
      end
    );
  end;
end;

procedure TOpenAi.Chat(const AMessages: TArray<IChatMessage>;
  AProc: TChatMessageEvent);
begin
  Chat(AMessages, nil, AProc);
end;

function TOpenAi.ContentToResponse(const AContent, AModelId: string;
  ADone: Boolean): TChatResponse;
begin
  Result := TChatResponse.Create;
  try
    Result.Model := AModelId;
    Result.CreatedAt := Now;
    Result.Done := ADone;
    Result.Message.Role := 'assistant';
    Result.Message.Content := AContent;
  except
    Result.Free;
    raise;
  end;
end;

function TOpenAi.ConvertMessages(
  const AMessages: TArray<IChatMessage>): TArray<TChatMessageBuild>;
var
  I: Integer;
begin
  SetLength(Result, Length(AMessages));
  for I := 0 to Length(AMessages) - 1 do
  begin
    if AMessages[I].Role = 'user' then
      Result[I] := TChatMessageBuild.User(AMessages[I].Content)
    else
      Result[I] := TChatMessageBuild.Assistant(AMessages[I].Content);
  end;
end;

constructor TOpenAi.Create;
begin
  inherited;
  FOpenAI := OpenAI.TOpenAI.Create();
  FOpenAI.Token := GetEnvironmentVariable('OPENAI_API_KEY');
  FModelId := LLM_MODEL_ID;
  FEmbeddingModelId := EMBEDDING_MODEL_ID;
  FToolCalls := [];
end;

destructor TOpenAi.Destroy;
begin
  inherited;
end;

function TOpenAi.GetApiKey: string;
begin
  Result := FOpenAI.Token;
end;

function TOpenAi.GetEmbeddingFromString(const AValue: string): TArray<Extended>;
var
  LEmbeddings: TEmbeddings;
begin
  LEmbeddings := FOpenAI.Embedding.Create(
    procedure (AParams: TEmbeddingParams)
    begin
      AParams.Model(FEmbeddingModelId);
      AParams.EncodingFormat(TEncodingFormat.Float);
      AParams.Input(AValue);
    end
  );
  try
    if Length(LEmbeddings.Data) < 1 then
      raise Exception.Create('Empty embedding result');

    Result := LEmbeddings.Data[0].Embedding;
  finally
    LEmbeddings.Free;
  end;
end;

function TOpenAi.GetEmbeddingFromStrings(
  const AValue: TArray<string>): TArray<TArray<Extended>>;
var
  LEmbeddings: TEmbeddings;
  I: Integer;
begin
  LEmbeddings := FOpenAI.Embedding.Create(
    procedure (AParams: TEmbeddingParams)
    begin
      AParams.Model(FEmbeddingModelId);
      AParams.EncodingFormat(TEncodingFormat.Float);
      AParams.Input(AValue);
    end
  );
  try
    if Length(LEmbeddings.Data) < 1 then
      raise Exception.Create('Empty embedding result');

    SetLength(Result, Length(LEmbeddings.Data));
    for I := Low(Result) to High(Result) do
      Result[I] := LEmbeddings.Data[I].Embedding;
  finally
    LEmbeddings.Free;
  end;
end;

function TOpenAi.GetEmbeddingModel: string;
begin
  Result := EMBEDDING_MODEL_ID;
end;

function TOpenAi.GetModel: string;
begin
  Result := LLM_MODEL_ID;
end;

function TOpenAi.GetModels: TArray<string>;
var
  LModels: TModels;
  I: Integer;
begin
  LModels := FOpenAI.Model.List;
  try
    SetLength(Result, Length(LModels.Data));
    for I := 0 to Length(LModels.Data) - 1 do
      Result[I] := LModels.Data[I].Id;
  finally
    LModels.Free;
  end;
end;

function TOpenAi.GetProvider: string;
begin
  Result := 'OpenAI';
end;

function TOpenAi.GetProxySettings: TProxySettings;
begin
  Result := FOpenAI.API.ProxySettings;
end;

function TOpenAi.GetSummary(const AContent: string): string;
begin
  Result := TAIEngine.GetSummary(Self, AContent);
end;

procedure TOpenAi.SetApiKey(const AValue: string);
begin
  FOpenAI.Token := AValue;
end;

procedure TOpenAi.SetProxySettings(const AProxySettings: TProxySettings);
begin
  FOpenAI.API.ProxySettings := AProxySettings;
end;

procedure TOpenAi.BuildToolsFromObject(AToolObject: TObject);
begin
  FTools := [];
  var LTools := TTools.CreateToolsSchema(AToolObject);
  try
    for var LTool in LTools do
    begin
      var LChatFunction: IChatFunction := TChatFunction.Create(AToolObject, LTool);
      FTools := FTools + [TChatToolFunctionParam.Create(LChatFunction)];
    end;
  finally
    LTools.Free;
  end;
end;

{ TChatFunction }

constructor TChatFunction.Create(AToolObject: TObject; AChatTool: TChatTool);
begin
  FToolObject := AToolObject;
  FChatTool := AChatTool;
end;

function TChatFunction.Execute(const Args: string): string;
begin
  raise Exception.Create('Execute');
end;

function TChatFunction.GetDescription: string;
begin
  Result := FChatTool.&Function.Description;
end;

function TChatFunction.GetName: string;
begin
  Result := FChatTool.&Function.Name;
end;

function TChatFunction.GetParameters: string;
begin
  Result := TNeon.ObjectToJSONString(FChatTool.&Function.Parameters, TNeonConfiguration.Camel);
end;

function TChatFunction.GetStrict: Boolean;
begin
  Result := True;
end;

{ TToolResponse }

constructor TToolResponse.Create(const AToolId, AContent: string);
begin
  ToolId := AToolId;
  Content := AContent;
end;

initialization

ServiceLocator.RegisterClass(TOpenAi);

end.
