unit Server.AiEngines.Ollama;

interface

uses
  System.Classes, System.SysUtils, System.JSON, System.Rtti,
  System.IOUtils, System.Generics.Collections,
  System.Net.HttpClient,
  System.net.URLClient,

  Dynamo.Core.Rtti,
  Neon.Core.Attributes,
  Neon.Core.Persistence,
  Neon.Core.Persistence.JSON,

  Server.AiEngines.Tools,
  Server.AiEngines.Core;

type
  EOllamaError = class(Exception);

  EOllamaHttpError = class(EOllamaError)
  private
    FStatusCode: Integer;
  public
    property StatusCode: Integer read FStatusCode;
    constructor Create(AResponse: IHTTPResponse);
  end;

  TEmbedRequest = class(TObject)
  private
    FModel: string;
    FInput: TArray<string>;
  public
    property Model: string read FModel write FModel;
    property Input: TArray<string> read FInput write FInput;
  end;

  TEmbedResponse = class(TObject)
  private
    FModel: string;
    FEmbeddings: TArray<TArray<Extended>>;
  public
    property Model: string read FModel write FModel;
    property Embeddings: TArray<TArray<Extended>> read FEmbeddings write FEmbeddings;
  end;

  TModel = class(TObject)
  private
    FName: string;
    FModifiedAt: string;
    FModel: string;
    FSize: Int64;
    FDigest: string;
  public
    property Name: string read FName write FName;
    property Model: string read FModel write FModel;
    property ModifiedAt: string read FModifiedAt write FModifiedAt;
    property Size: Int64 read FSize write FSize;
    property Digest: string read FDigest write FDigest;
  end;

  TModelList = class(TObjectList<TModel>)
  end;

  TModelCollection = class(TObject)
  private
    FModels: TModelList;
  public
    property Models: TModelList read FModels write FModels;

    constructor Create;
    destructor Destroy; override;
  end;

  TOllamaMessage = class(TObject)
  private
    FRole: string;
    FId: Integer;
    FContent: string;
    FToolCalls: TChatToolCalls;
  public
    constructor Create;
    destructor Destroy; override;

    property Id: Integer read FId write FId;
    property Role: string read FRole write FRole;
    property Content: string read FContent write FContent;
    [NeonProperty('tool_calls')]
    property ToolCalls: TChatToolCalls read FToolCalls write FToolCalls;
  end;

  TOllamaMessageList = class(TObjectList<TOllamaMessage>)
  public
    procedure AddMessage(const ARole, AContent: string; AToolCalls: TChatToolCalls = nil);
  end;

  TOllamaChat = class(TObject)
  private
    FMessages: TOllamaMessageList;
    FModel: string;
    FStream: Boolean;
    FTools: TChatTools;
  public
    property Model: string read FModel write FModel;
    property Messages: TOllamaMessageList read FMessages write FMessages;
    property Stream: Boolean read FStream write FStream;
    property Tools: TChatTools read FTools write FTools;

    constructor Create;
    destructor Destroy; override;
  end;

  [Alias('ollama')]
  TOllamaAi = class(TInterfacedObject, IAIEngine, IEmbedder)
  private
    //FSerializer: TJsonSerializer;
    FHttpClient: THTTPClient;
    FResponseProc: TChatMessageEvent;
    FOldChunk: string;
    FToolObject: TObject;
    FToolResponses: TArray<string>;
    FToolCalls: TChatToolCalls;
    FDone: Boolean;
    procedure HandleChunkReceiveDataEx(const Sender: TObject; AContentLength, AReadCount: Int64; AChunk: Pointer; AChunkLength: Cardinal; var AAbort: Boolean);
    procedure HandleToolsCall(const AToolCalls: TChatToolCalls);
    function CreateDoneReponse: TChatResponse;
    procedure OllamaChat(AOllamaChat: TOllamaChat; AProc: TChatMessageEvent);
  public
    function GetApiKey: string;
    procedure SetApiKey(const AValue: string);
    procedure Chat(const AMessages: TArray<IChatMessage>; AProc: TChatMessageEvent); overload;
    procedure Chat(const AMessages: TArray<IChatMessage>; AToolObject: TObject; AProc: TChatMessageEvent); overload;
    function GetProvider: string;
    function GetModels: TArray<string>;
    function GetSummary(const AContent: string): string;
    function GetProxySettings: TProxySettings;
    procedure SetProxySettings(const AProxySettings: TProxySettings);
    function GetModel: string;

    function GetEmbeddingModel: string;
    function GetEmbeddingFromString(const AValue: string): TArray<Extended>;
    function GetEmbeddingFromStrings(const AValue: TArray<string>): TArray<TArray<Extended>>;

    constructor Create;
    destructor Destroy; override;
  end;

implementation

uses
  Dynamo.Core.ServiceLocator;

const
  OllamaUrl = 'http://localhost:11434';
  LLM_MODEL_ID = 'llama3.2';
  EMBEDDING_MODEL_ID = 'all-minilm';


{ TOllamaAi }

procedure TOllamaAi.OllamaChat(AOllamaChat: TOllamaChat; AProc: TChatMessageEvent);
begin
  FOldChunk := '';
  var LRequestStream := TMemoryStream.Create;
  try
    TNeon.ObjectToStream(AOllamaChat, LRequestStream, TNeonConfiguration.Camel);
    LRequestStream.Position := 0;

    FHttpClient.OnReceiveDataEx := HandleChunkReceiveDataEx;

    var LResponse := FHttpClient.Post(OllamaUrl + '/api/chat', LRequestStream);
    if LResponse.StatusCode <> 200 then
      raise EOllamaHttpError.Create(LResponse);

  finally
    LRequestStream.Free;
  end;

  if not FDone then
  begin
    var LDoneResponse := CreateDoneReponse;
    try
      AProc(LDoneResponse);
    finally
      LDoneResponse.Free;
    end;
  end;
end;

procedure TOllamaAi.Chat(const AMessages: TArray<IChatMessage>;
  AToolObject: TObject; AProc: TChatMessageEvent);
begin
  FOldChunk := '';
  FToolObject := AToolObject;
  FDone := False;
  FResponseProc := AProc;
  FToolResponses := [];
  var LOllamaChat := TOllamaChat.Create;
  try
    LOllamaChat.Model := LLM_MODEL_ID;
    for var LMessage in AMessages do
    begin
      LOllamaChat.Messages.AddMessage(LMessage.Role, LMessage.Content);
    end;

    if Assigned(AToolObject) then
    begin
      var LTools := TTools.CreateToolsSchema(AToolObject);
      LOllamaChat.Tools := LTools;
    end;

    OllamaChat(LOllamaChat, AProc);

    if Length(FToolResponses) > 0 then
    begin
      var LToolResponses := FToolResponses;
      // Avoid a possible infinite loop
      FToolResponses := [];
      var LTools := TChatToolCalls.Create;
      LTools.Assign(FToolCalls);

      LOllamaChat.Messages.AddMessage('assistant', '', LTools);
      for var LResponse in LToolResponses do
      begin
        LOllamaChat.Messages.AddMessage('tool', LResponse);
      end;
      OllamaChat(LOllamaChat, AProc);

    end;

  finally
    LOllamaChat.Free;
  end;
end;

procedure TOllamaAi.Chat(const AMessages: TArray<IChatMessage>;
  AProc: TChatMessageEvent);
begin
  Chat(AMessages, nil, AProc);
end;

constructor TOllamaAi.Create;
begin
  FHttpClient := THTTPClient.Create;
  FToolCalls := TChatToolCalls.Create;
end;

function TOllamaAi.CreateDoneReponse: TChatResponse;
begin
  Result := TChatResponse.Create;
  try
    Result.Model := LLM_MODEL_ID;
    Result.CreatedAt := Now;
    Result.Message.Role := 'Assistant';
    Result.Message.Content := '';
  except
    Result.Free;
    raise;
  end;
end;

destructor TOllamaAi.Destroy;
begin
  FHttpClient.Free;
  FToolCalls.Free;
  inherited;
end;

function TOllamaAi.GetApiKey: string;
begin

end;

function TOllamaAi.GetEmbeddingFromString(
  const AValue: string): TArray<Extended>;
begin
  Result := GetEmbeddingFromStrings([AValue])[0];
end;

function TOllamaAi.GetEmbeddingFromStrings(
  const AValue: TArray<string>): TArray<TArray<Extended>>;
begin
  var LRequest := TEmbedRequest.Create;
  try
    LRequest.Model := EMBEDDING_MODEL_ID;
    LRequest.Input := AValue;

    var LRequestStream := TMemoryStream.Create;
    try
      TNeon.ObjectToStream(LRequest, LRequestStream, TNeonConfiguration.Camel);
      LRequestStream.Position := 0;
      var LResponse := FHttpClient.Post(OllamaUrl + '/api/embed', LRequestStream);
      if LResponse.StatusCode <> 200 then
        raise EOllamaHttpError.Create(LResponse);

      var LEmbedResponse := TNeon.JSONToObject<TEmbedResponse>(LResponse.ContentAsString(), TNeonConfiguration.Camel);
      try
        Result := LEmbedResponse.Embeddings;
      finally
        LEmbedResponse.Free;
      end;
    finally
      LRequestStream.Free;
    end;

  finally
    LRequest.Free;
  end;
end;

function TOllamaAi.GetEmbeddingModel: string;
begin
  Result := EMBEDDING_MODEL_ID;
end;

function TOllamaAi.GetModel: string;
begin
  Result := LLM_MODEL_ID;
end;

function TOllamaAi.GetModels: TArray<string>;
begin
  var LResponse := FHttpClient.Get(OllamaUrl + '/api/tags');
  if LResponse.StatusCode <> 200 then
    raise EOllamaHttpError.Create(LResponse);

  Result := [];
  var LModelCollection := TNeon.JSONToObject<TModelCollection>(LResponse.ContentAsString(), TNeonConfiguration.Camel);
  try
    for var LModel in LModelCollection.Models do
      Result := Result + [LModel.Name];
  finally
    LModelCollection.Free;
  end;
end;

function TOllamaAi.GetProvider: string;
begin
  Result := 'Ollama';
end;

function TOllamaAi.GetProxySettings: TProxySettings;
begin
  Result := FHttpClient.ProxySettings;
end;

function TOllamaAi.GetSummary(const AContent: string): string;
begin
  Result := TAIEngine.GetSummary(Self, AContent);
end;

procedure TOllamaAi.HandleChunkReceiveDataEx(const Sender: TObject;
  AContentLength, AReadCount: Int64; AChunk: Pointer; AChunkLength: Cardinal;
  var AAbort: Boolean);
var
  LBytes: TBytes;
  LChunkStr: string;
begin
  if AChunkLength > 0 then
  begin
    SetLength(LBytes, AChunkLength);
    Move(AChunk^, LBytes[0], AChunkLength);
    LChunkStr := TEncoding.UTF8.GetString(LBytes);

    try
      var LResponse := TNeon.JSONToObject<TChatResponse>(FOldChunk + LChunkStr, TNeonConfiguration.Camel);
      try
        FOldChunk := '';
        FDone := LResponse.Done;
        if LResponse.Message.ToolCalls.Count > 0 then
          HandleToolsCall(LResponse.Message.ToolCalls)
        else if Assigned(FResponseProc) then
          FResponseProc(LResponse);
      finally
        LResponse.Free;
      end;
    except
      on EJSONParseException do
      begin
        FOldChunk := FOldChunk + LChunkStr;
      end;
    end;
  end;
end;

procedure TOllamaAi.HandleToolsCall(const AToolCalls: TChatToolCalls);

  function GetIndexOfParam(AMethodRtti: TRttiMethod; const AParamName: string): Integer;
  begin
    var Index := 0;
    for var LParam in AMethodRtti.GetParameters do
    begin
      if SameText(LParam.Name, AParamName) then
        Exit(Index);
      Inc(Index);
    end;
    raise EOllamaError.CreateFmt('Parameter "%s" not found', [AParamName]);
  end;

begin
  FToolResponses := [];
  var LRttiContext := TRttiContext.Create;
  try
    for var LToolCall in AToolCalls do
    begin
      var LFunctionName := LToolCall.&Function.Name;
      var LToolRtti := LRttiContext.GetType(FToolObject.ClassType);
      var LMethodRtti := LToolRtti.GetMethod(LFunctionName);
      var LArgs :TArray<TValue>;
      SetLength(LArgs, Length(LMethodRtti.GetParameters));
      for var LArgPair in LToolCall.&Function.Arguments do
      begin
        LArgs[GetIndexOfParam(LMethodRtti, LArgPair.Key)] := LArgPair.Value;
      end;

      var LResult := LMethodRtti.Invoke(FToolObject, LArgs);
      FToolResponses := FToolResponses + [LResult.ToString];
      FToolCalls.Assign(AToolCalls);
    end;
  finally
    LRttiContext.Free;
  end;
end;

procedure TOllamaAi.SetApiKey(const AValue: string);
begin

end;

procedure TOllamaAi.SetProxySettings(const AProxySettings: TProxySettings);
begin
  FHttpClient.ProxySettings := AProxySettings;
end;

{ EOllamaHttpError }

constructor EOllamaHttpError.Create(AResponse: IHTTPResponse);
begin
  inherited Create(IntToStr(AResponse.StatusCode) + ' ' + AResponse.StatusText);
  FStatusCode := AResponse.StatusCode;
end;

{ TModelCollection }

constructor TModelCollection.Create;
begin
  inherited Create;
  FModels := TModelList.Create
end;

destructor TModelCollection.Destroy;
begin
  FModels.Free;
  inherited;
end;

{ TOllamaChat }

constructor TOllamaChat.Create;
begin
  inherited Create;
  FMessages := TOllamaMessageList.Create;
  FTools := nil;
end;

destructor TOllamaChat.Destroy;
begin
  FMessages.Free;
  if Assigned(FTools) then
    FTools.Free;
  inherited;
end;

{ TOllamaMessageList }

procedure TOllamaMessageList.AddMessage(const ARole, AContent: string; AToolCalls: TChatToolCalls = nil);
begin
  var LMessage := TOllamaMessage.Create;
  try
    LMessage.Id := Count + 1;
    LMessage.Role := ARole;
    LMessage.Content := AContent;
    LMessage.ToolCalls := AToolCalls;
    Add(LMessage);
  except
    LMessage.Free;
    raise;
  end;
end;

{ TOllamaMessage }

constructor TOllamaMessage.Create;
begin
  FToolCalls := nil;
end;

destructor TOllamaMessage.Destroy;
begin
  if Assigned(FToolCalls) then
    FToolCalls.Free;
  inherited;
end;

initialization

ServiceLocator.RegisterClass(TOllamaAi);

end.
