unit Server.AiEngines.MistralAI;

interface

uses
  System.Classes, System.SysUtils, System.net.URLClient,

  MistralAI,
  MistralAI.Types,

  Dynamo.Core.Rtti,

  Server.AiEngines.Core;

type
  [Alias('mistral')]
  TMistralAi = class(TInterfacedObject, IAIEngine)
  private
    FMistralAI: IMistralAI;
    FModelId: string;
    function ConvertMessages(const AMessages: TArray<IChatMessage>): TArray<TChatMessagePayload>;
    function ContentToResponse(const AContent, AModelId: string;
      ADone: Boolean): TChatResponse;
  public
    procedure Chat(const AMessages: TArray<IChatMessage>; AProc: TChatMessageEvent); overload;
    procedure Chat(const AMessages: TArray<IChatMessage>; AToolObject: TObject; AProc: TChatMessageEvent); overload;
    function GetProvider: string;
    function GetModels: TArray<string>;
    function GetSummary(const AContent: string): string;
    function GetApiKey: string;
    procedure SetApiKey(const AValue: string);
    function GetProxySettings: TProxySettings;
    procedure SetProxySettings(const AProxySettings: TProxySettings);
    function GetModel: string;

    constructor Create;
    destructor Destroy; override;
  end;

implementation

{ TMistralAi }

uses
  Dynamo.Core.ServiceLocator;

const
  LLM_MODEL_ID = 'mistral-large-latest';

procedure TMistralAi.Chat(const AMessages: TArray<IChatMessage>;
  AToolObject: TObject; AProc: TChatMessageEvent);
begin
  raise EAiEngineToolsError.Create('Not yet implemented');
end;

function TMistralAi.ContentToResponse(const AContent, AModelId: string; ADone: Boolean): TChatResponse;
begin
  Result := TChatResponse.Create;
  try
    Result.Model := AModelId;
    Result.CreatedAt := Now;
    Result.Done := ADone;
    Result.Message.Role := 'assistant';
    Result.Message.Content := AContent;
  except
    Result.Free;
    raise;
  end;
end;


procedure TMistralAi.Chat(const AMessages: TArray<IChatMessage>; AProc: TChatMessageEvent);
begin
  FMistralAI.Chat.CreateStream(
    procedure(Params: TChatParams)
    begin
      Params.Model(FModelId);
      Params.Messages(ConvertMessages(AMessages));
      Params.MaxTokens(1024);
      Params.Stream;
    end,
    procedure(var Chat: TChat; IsDone: Boolean; var Cancel: Boolean)

      procedure SendResponse(const AContent: string; ADone: Boolean);
      var
        LResponse: TChatResponse;
      begin
        LResponse := ContentToResponse(AContent, FModelId, ADone);
        try
          AProc(LResponse);
        finally
          LResponse.Free;
        end;
      end;

    begin
      if (not IsDone) and Assigned(Chat) then
        begin
          if Chat.Choices[0].Delta.Content <> '' then
            SendResponse(Chat.Choices[0].Delta.Content, IsDone);
        end;
      if IsDone then
        SendResponse('', IsDone);
    end);

end;

function TMistralAi.ConvertMessages(
  const AMessages: TArray<IChatMessage>): TArray<TChatMessagePayload>;
var
  I: Integer;
begin
  SetLength(Result, Length(AMessages));
  for I := 0 to Length(AMessages) - 1 do
  begin
    if AMessages[I].Role = 'user' then
      Result[I] := Payload.User(AMessages[I].Content)
    else
      Result[I] := Payload.Assistant(AMessages[I].Content);
  end;
end;

constructor TMistralAi.Create;
begin
  inherited;
  FMistralAI := TMistralAIFactory.CreateInstance('');
  FMistralAI.Token := GetEnvironmentVariable('MISTRALAI_API_KEY');
  FModelId := LLM_MODEL_ID;
end;

destructor TMistralAi.Destroy;
begin
  inherited;
end;

function TMistralAi.GetApiKey: string;
begin
  Result := FMistralAI.Token;
end;

function TMistralAi.GetModel: string;
begin
  Result := LLM_MODEL_ID;
end;

function TMistralAi.GetModels: TArray<string>;
var
  LModels: TModels;
  I: Integer;
begin
  LModels := FMistralAI.Models.List;
  try
    SetLength(Result, Length(LModels.Data));
    for I := 0 to Length(LModels.Data) - 1 do
      Result[I] := LModels.Data[I].Id;
  finally
    LModels.Free;
  end;
end;

function TMistralAi.GetProvider: string;
begin
  Result := 'Mistral';
end;

function TMistralAi.GetProxySettings: TProxySettings;
begin

end;

function TMistralAi.GetSummary(const AContent: string): string;
begin
  Result := TAIEngine.GetSummary(Self, AContent);
end;

procedure TMistralAi.SetApiKey(const AValue: string);
begin
  FMistralAI.Token := AValue;
end;

procedure TMistralAi.SetProxySettings(const AProxySettings: TProxySettings);
begin

end;

initialization

ServiceLocator.RegisterClass(TMistralAi);

end.

