unit Server.AiEngines.GeminiAI;

interface

uses
  System.Classes, System.SysUtils, System.Net.URLClient,

  Gemini,
  Gemini.Chat,
  Gemini.Models,
  Gemini.Embeddings,

  Dynamo.Core.Rtti,

  Server.AiEngines.Core;

type
  [Alias('geminiai')]
  TGeminiAi = class(TInterfacedObject, IAIEngine, IEmbedder)
  private
    FGemini: IGemini;
    FModelId: string;
    FEmbeddingModelId: string;
    function ConvertMessages(const AMessages: TArray<IChatMessage>): TArray<TContentPayload>;
    function ContentToResponse(const AContent, AModelId: string;
      ADone: Boolean): TChatResponse;
    function GetParams(const AValue: TArray<string>; AStartIndex, ACount: Integer): TArray<TEmbeddingRequestParams>;
  public
    procedure Chat(const AMessages: TArray<IChatMessage>; AProc: TChatMessageEvent); overload;
    procedure Chat(const AMessages: TArray<IChatMessage>; AToolObject: TObject; AProc: TChatMessageEvent); overload;
    function GetProvider: string;
    function GetModels: TArray<string>;
    function GetSummary(const AContent: string): string;
    function GetApiKey: string;
    procedure SetApiKey(const AValue: string);
    function GetProxySettings: TProxySettings;
    procedure SetProxySettings(const AProxySettings: TProxySettings);
    function GetModel: string;

    function GetEmbeddingModel: string;
    function GetEmbeddingFromString(const AValue: string): TArray<Extended>;
    function GetEmbeddingFromStrings(const AValue: TArray<string>): TArray<TArray<Extended>>;

    constructor Create;
    destructor Destroy; override;
  end;

implementation

{ TGeminiAi }

uses
  Dynamo.Core.ServiceLocator;

const
  LLM_MODEL_ID = 'models/gemini-2.0-flash';
  EMBEDDING_MODEL_ID = 'models/text-embedding-004';

function Min(v1, v2: Integer): Integer;
begin
  if v1 < v2 then
    Result := v1
  else
    Result := v2;
end;

procedure TGeminiAi.Chat(const AMessages: TArray<IChatMessage>;
  AProc: TChatMessageEvent);
begin
  FGemini.Chat.CreateStream(FModelId,
    procedure(Params: TChatParams)
    begin
      Params.Contents(ConvertMessages(AMessages));
      //Params.MaxTokens(1024);
    end,
    procedure(var Chat: TChat; IsDone: Boolean; var Cancel: Boolean)

      procedure SendResponse(const AContent: string; ADone: Boolean);
      var
        LResponse: TChatResponse;
      begin
        LResponse := ContentToResponse(AContent, FModelId, ADone);
        try
          AProc(LResponse);
        finally
          LResponse.Free;
        end;
      end;

    var
      LCandidate: TChatCandidate;
      LChatPart: TChatPart;

    begin
      if (not IsDone) and Assigned(Chat) then
      begin
        for LCandidate in Chat.Candidates do
        begin
          if LCandidate.FinishReason <> TFinishReason.SAFETY then
          begin
            for LChatPart in LCandidate.Content.Parts do
              if Assigned(LChatPart) then
                SendResponse(LChatPart.Text, IsDone);
          end;
        end;
      end
      else if IsDone then
        SendResponse('', IsDone);
    end
  );
end;


procedure TGeminiAi.Chat(const AMessages: TArray<IChatMessage>;
  AToolObject: TObject; AProc: TChatMessageEvent);
begin
  raise EAiEngineToolsError.Create('Not yet implemented');
end;

function TGeminiAi.ContentToResponse(const AContent, AModelId: string;
  ADone: Boolean): TChatResponse;
begin
  Result := TChatResponse.Create;
  try
    Result.Model := AModelId;
    Result.CreatedAt := Now;
    Result.Done := ADone;
    Result.Message.Role := 'assistant';
    Result.Message.Content := AContent;
  except
    Result.Free;
    raise;
  end;
end;

function TGeminiAi.ConvertMessages(
  const AMessages: TArray<IChatMessage>): TArray<TContentPayload>;
var
  I: Integer;
begin
  SetLength(Result, Length(AMessages));
  for I := 0 to Length(AMessages) - 1 do
  begin
    if AMessages[I].Role = 'user' then
      Result[I] := TPayload.User(AMessages[I].Content)
    else
      Result[I] := TPayload.Assistant(AMessages[I].Content);
  end;
end;

constructor TGeminiAi.Create;
begin
  inherited;
  FGemini := TGeminiFactory.CreateInstance('');
  FGemini.Token := GetEnvironmentVariable('GEMINI_API_KEY');

  FModelId := LLM_MODEL_ID;
  FEmbeddingModelId := EMBEDDING_MODEL_ID;
end;

destructor TGeminiAi.Destroy;
begin
  FGemini := nil;
  inherited;
end;

function TGeminiAi.GetApiKey: string;
begin
  Result := FGemini.Token;
end;

function TGeminiAi.GetEmbeddingFromString(const AValue: string): TArray<Extended>;
var
  LEmbedding: TEmbedding;
begin
  LEmbedding := FGemini.Embeddings.Create(FEmbeddingModelId,
    procedure (Params: TEmbeddingParams)
    begin
      Params.Content([AValue]);
    end);
  try
    Result := DoubleArrayToExtended(LEmbedding.Embedding.Values);
  finally
    LEmbedding.Free;
  end;
end;

function TGeminiAi.GetParams(const AValue: TArray<string>; AStartIndex, ACount: Integer): TArray<TEmbeddingRequestParams>;
var
  I: Integer;
  Len: Integer;
begin
  Len := Min(Length(AValue), ACount + AStartIndex);
  SetLength(Result, Len - AStartIndex);
  for I := AStartIndex to Len - 1 do
  begin
    Result[I - AStartIndex] :=
      TEmbeddingRequestParams.Create(
        procedure (var Params: TEmbeddingRequestParams)
        begin
          Params.Content([AValue[I]]);
        end);
  end;
end;

function TGeminiAi.GetEmbeddingFromStrings(
  const AValue: TArray<string>): TArray<TArray<Extended>>;
const
  MaxLines = 99;
var
  LEmbeddings: TEmbeddings;
  I: Integer;
  StartIndex: Integer;
begin
  SetLength(Result, Length(AValue));
  StartIndex := 0;
  repeat
    LEmbeddings := FGemini.Embeddings.CreateBatch(FEmbeddingModelId,
      procedure (Parameters: TEmbeddingBatchParams)
      begin
        Parameters.Requests(GetParams(AValue, StartIndex, MaxLines));
      end);

    try
      //SetLength(Result, Length(LEmbeddings.Embeddings));
      for I := 0 to Length(LEmbeddings.Embeddings) - 1 do
        Result[I + StartIndex] := DoubleArrayToExtended(LEmbeddings.Embeddings[I].Values);
    finally
      LEmbeddings.Free;
    end;
    StartIndex := StartIndex + MaxLines;
  until StartIndex >= Length(AValue);
end;

function TGeminiAi.GetEmbeddingModel: string;
begin
  Result := EMBEDDING_MODEL_ID;
end;

function TGeminiAi.GetModel: string;
begin
  Result := LLM_MODEL_ID;
end;

function TGeminiAi.GetModels: TArray<string>;
var
  LModels: TModels;
  I: Integer;
begin
  LModels := FGemini.Models.List;
  try
    SetLength(Result, Length(LModels.Models));
    for I := 0 to Length(LModels.Models) - 1 do
      Result[I] := LModels.Models[I].Name;
  finally
    LModels.Free;
  end;
end;

function TGeminiAi.GetProvider: string;
begin
  Result := 'GeminiAI';
end;

function TGeminiAi.GetProxySettings: TProxySettings;
begin

end;

function TGeminiAi.GetSummary(const AContent: string): string;
begin
  Result := TAIEngine.GetSummary(Self, AContent);
end;

procedure TGeminiAi.SetApiKey(const AValue: string);
begin
  FGemini.Token := AValue;
end;

procedure TGeminiAi.SetProxySettings(const AProxySettings: TProxySettings);
begin

end;

initialization

ServiceLocator.RegisterClass(TGeminiAi);

end.

