unit Server.AiEngines.Core;

interface

uses
  System.Classes, System.SysUtils, System.net.URLClient, System.Generics.Collections,
  Neon.Core.Attributes;

type
  EAiEngineError = class(Exception);

  EAiEngineToolsError = class(EAiEngineError);

  ChatToolAttribute = class(TCustomAttribute)
  private
    FDescription: string;
  public
    constructor Create(const ADescription: string);

    property Description: string read FDescription;
  end;

  TChatFunctionArgsCall = class(TDictionary<string, string>)
  end;

  TChatFunctionCall = class
  private
    FName: string;
    FArguments: TChatFunctionArgsCall;
  public
    procedure Assign(AFunction: TChatFunctionCall);

    constructor Create;
    destructor Destroy; override;

    property Name: string read FName write FName;
    property Arguments: TChatFunctionArgsCall read FArguments write FArguments;
  end;

  TChatToolCall = class
  private
    FFunction: TChatFunctionCall;
  public
    procedure Assign(AToolCall: TChatToolCall);

    constructor Create;
    destructor Destroy; override;

    property &Function: TChatFunctionCall read FFunction write FFunction;
  end;

  TChatToolCalls = class(TObjectList<TChatToolCall>)
  public
    procedure Assign(AToolCalls: TChatToolCalls);
  end;

  TChatResponseMessage = class
  private
    FRole: string;
    FContent: string;
    FToolCalls: TChatToolCalls;
  public
    constructor Create;
    destructor Destroy; override;

    property Content: string read FContent write FContent;
    property Role: string read FRole write FRole;
    [NeonProperty('tool_calls')]
    property ToolCalls: TChatToolCalls read FToolCalls write FToolCalls;
  end;

  TChatResponse = class
  private
    FModel: string;
    FCreatedAt: TDateTime;
    FDone: Boolean;
    FMessage: TChatResponseMessage;
  public
    property Model: string read FModel write FModel;
    property CreatedAt: TDateTime read FCreatedAt write FCreatedAt;
    property &Message: TChatResponseMessage read FMessage write FMessage;
    property Done: Boolean read FDone write FDone;

    constructor Create;
    destructor Destroy; override;
  end;

  TChatMessageEvent = reference to procedure (AChat: TChatResponse);

  IChatMessage = interface
    ['{184CA902-5944-48DA-BA2E-5A175441FEBD}']
    function GetRole: string;
    function GetContent: string;

    property Role: string read GetRole;
    property Content: string read GetContent;
  end;

  IAIEngine = interface
    ['{855ADDC4-ED21-40F4-8A61-FF61DF4D2E37}']
    function GetProvider: string;
    procedure Chat(const AMessages: TArray<IChatMessage>; AProc: TChatMessageEvent); overload;
    procedure Chat(const AMessages: TArray<IChatMessage>; AToolObject: TObject; AProc: TChatMessageEvent); overload;
    function GetModels: TArray<string>;
    function GetSummary(const AContent: string): string;
    function GetModel: string;
    function GetApiKey: string;
    procedure SetApiKey(const AValue: string);
    function GetProxySettings: TProxySettings;
    procedure SetProxySettings(const AProxySettings: TProxySettings);

    property ApiKey: string read GetApiKey write SetApiKey;
    property ProxySettings: TProxySettings read GetProxySettings write SetProxySettings;
    property Provider: string read GetProvider;
    property Model: string read GetModel;
  end;

  IEmbedder = interface
    ['{61DAB5DC-F2A2-482F-AFF8-97ADAA9E4B92}']
    function GetProvider: string;
    function GetEmbeddingFromString(const AValue: string): TArray<Extended>;
    function GetEmbeddingFromStrings(const AValue: TArray<string>): TArray<TArray<Extended>>;
    function GetEmbeddingModel: string;
    function GetApiKey: string;

    property Provider: string read GetProvider;
    property EmbeddingModel: string read GetEmbeddingModel;
    procedure SetApiKey(const AValue: string);
    property ApiKey: string read GetApiKey write SetApiKey;
  end;

  TAIEngine = class
  public
    class function GetSummary(LLM: IAIEngine; const AContent: string): string; static;
    class function CreateUserPayload(const AContent: string): IChatMessage; static;
    class function CreateAssistantPayload(const AContent: string): IChatMessage; static;
  end;

  TChatMessage = class(TInterfacedObject, IChatMessage)
  private
    FRole: string;
    FContent: string;
  private
    function GetRole: string;
    function GetContent: string;
    constructor Create(const ARole, AContent: string);
  end;

function DoubleArrayToExtended(const AValues: TArray<Double>): TArray<Extended>;
function EmbeddingToStr(const AEmbedding: TArray<Extended>): string;

implementation

function DoubleArrayToExtended(const AValues: TArray<Double>): TArray<Extended>;
begin
  SetLength(Result, Length(AValues));
  for var I := 0 to Length(AValues) - 1 do
  begin
    Result[I] := AValues[I];
  end;
end;

function EmbeddingToStr(const AEmbedding: TArray<Extended>): string;
begin
  var LStrArray: TArray<string> := [];
  SetLength(LStrArray, Length(AEmbedding));
  SetLength(Result, Length(AEmbedding));
  for var I := 0 to Length(AEmbedding) - 1 do
  begin
    LStrArray[I] := FloatToStr(AEmbedding[I]);
  end;
  Result := string.Join(', ', LStrArray);
end;

{ TAIEngine }

class function TAIEngine.CreateAssistantPayload(const AContent: string): IChatMessage;
begin
  Result := TChatMessage.Create('assistant', AContent);
end;

class function TAIEngine.CreateUserPayload(const AContent: string): IChatMessage;
begin
  Result := TChatMessage.Create('user', AContent);
end;

class function TAIEngine.GetSummary(LLM: IAIEngine; const AContent: string): string;
const
  PromptTpl = 'Fammi un breve riassunto in italiano (massimo 100 parole) del documento che segue. Il riassunto deve partire dopo tre trattini --- ' + sLineBreak + '%s';
var
  LSummary: string;
  LPrompt: TArray<IChatMessage>;
  LPos: Integer;
begin
  LSummary := '';

  LPrompt := [TAIEngine.CreateUserPayload(Format(PromptTpl, [AContent]))];

  LLM.Chat(LPrompt,
    procedure(AChat: TChatResponse)
    begin
      if not AChat.Done then
      begin
        LSummary := LSummary + AChat.Message.Content;
      end
    end);

  LPos := Pos('---', LSummary);
  if LPos < 1 then
    Exit(LSummary);

  Result := Copy(LSummary, LPos + 3, Length(LSummary));
end;


{ TChatResponse }

constructor TChatResponse.Create;
begin
  inherited;
  FMessage := TChatResponseMessage.Create;
end;

destructor TChatResponse.Destroy;
begin
  FMessage.Free;
  inherited;
end;

{ TChatMessage }

constructor TChatMessage.Create(const ARole, AContent: string);
begin
  inherited Create;
  FRole := ARole;
  FContent := AContent;
end;

function TChatMessage.GetContent: string;
begin
  Result := FContent;
end;

function TChatMessage.GetRole: string;
begin
  Result := FRole;
end;

{ ChatToolAttribute }

constructor ChatToolAttribute.Create(const ADescription: string);
begin
  FDescription := ADescription;
end;

{ TChatResponseMessage }

constructor TChatResponseMessage.Create;
begin
  FToolCalls := TChatToolCalls.Create;
end;

destructor TChatResponseMessage.Destroy;
begin
  FToolCalls.Free;
  inherited;
end;

{ TChatToolCall }

procedure TChatToolCall.Assign(AToolCall: TChatToolCall);
begin
  &Function.Assign(AToolCall.&Function);
end;

constructor TChatToolCall.Create;
begin
  FFunction := TChatFunctionCall.Create;
end;

destructor TChatToolCall.Destroy;
begin
  FFunction.Free;
  inherited;
end;

{ TChatFunctionCall }

procedure TChatFunctionCall.Assign(AFunction: TChatFunctionCall);
begin
  Name := AFunction.Name;
  for var LSourceArgPair in AFunction.Arguments do
  begin
    Arguments.Add(LSourceArgPair.Key, LSourceArgPair.Value);
  end;
end;

constructor TChatFunctionCall.Create;
begin
  FArguments := TChatFunctionArgsCall.Create;
end;

destructor TChatFunctionCall.Destroy;
begin
  FArguments.Free;
  inherited;
end;

{ TChatToolCalls }

procedure TChatToolCalls.Assign(AToolCalls: TChatToolCalls);
begin
  Clear;
  for var LSourceToolCall in AToolCalls do
  begin
    var LToolCall := TChatToolCall.Create;
    LToolCall.Assign(LSourceToolCall);
    Add(LToolCall);
  end;
end;

end.

