unit Server.AiEngines.Anthropic;

interface

uses
  System.Classes, System.SysUtils, System.net.URLClient,

  Anthropic,
  Anthropic.Chat,
  Anthropic.Models,

  Dynamo.Core.Rtti,

  Server.AiEngines.Core;

type
  [Alias('anthropic')]
  TAnthropicAi = class(TInterfacedObject, IAIEngine)
  private
    FAnthropicAI: IAnthropic;
    FModelId: string;
    function ConvertMessages(const AMessages: TArray<IChatMessage>): TArray<TChatMessagePayload>;
    function ContentToResponse(const AContent, AModelId: string;
      ADone: Boolean): TChatResponse;
  public
    function GetApiKey: string;
    procedure SetApiKey(const AValue: string);
    function GetProxySettings: TProxySettings;
    procedure SetProxySettings(const AProxySettings: TProxySettings);
    procedure Chat(const AMessages: TArray<IChatMessage>; AProc: TChatMessageEvent); overload;
    procedure Chat(const AMessages: TArray<IChatMessage>; AToolObject: TObject; AProc: TChatMessageEvent); overload;
    function GetProvider: string;
    function GetModels: TArray<string>;
    function GetSummary(const AContent: string): string;
    function GetModel: string;

    constructor Create;
    destructor Destroy; override;
  end;

implementation

{ TAnthropicAi }

uses
  Dynamo.Core.ServiceLocator;

const
  LLM_MODEL_ID = 'claude-sonnet-4-5-20250929';

procedure TAnthropicAi.Chat(const AMessages: TArray<IChatMessage>;
  AProc: TChatMessageEvent);
begin
  FAnthropicAI.Chat.CreateStream(
    procedure(Params: TChatParams)
    begin
      Params.Model(FModelId);
      Params.Messages(ConvertMessages(AMessages));
      Params.MaxTokens(8192);
      Params.Stream;
    end,
    procedure(var Chat: TChat; IsDone: Boolean; var Cancel: Boolean)

      procedure SendResponse(const AContent: string; ADone: Boolean);
      var
        LResponse: TChatResponse;
      begin
        LResponse := ContentToResponse(AContent, FModelId, ADone);
        try
          AProc(LResponse);
        finally
          LResponse.Free;
        end;
      end;

    begin
      if (not IsDone) and Assigned(Chat) then
      begin
        if Chat.Delta.Text <> '' then
        begin
          SendResponse(Chat.Delta.Text, IsDone);
        end;
      end
      else if IsDone then
        SendResponse('', IsDone);
    end
  );
end;

procedure TAnthropicAi.Chat(const AMessages: TArray<IChatMessage>;
  AToolObject: TObject; AProc: TChatMessageEvent);
begin
  raise EAiEngineToolsError.Create('Not yet implemented');
end;

function TAnthropicAi.ContentToResponse(const AContent, AModelId: string;
  ADone: Boolean): TChatResponse;
begin
  Result := TChatResponse.Create;
  try
    Result.Model := AModelId;
    Result.CreatedAt := Now;
    Result.Done := ADone;
    Result.Message.Role := 'assistant';
    Result.Message.Content := AContent;
  except
    Result.Free;
    raise;
  end;
end;

function TAnthropicAi.ConvertMessages(
  const AMessages: TArray<IChatMessage>): TArray<TChatMessagePayload>;
var
  I: Integer;
begin
  SetLength(Result, Length(AMessages));
  for I := 0 to Length(AMessages) - 1 do
  begin
    if AMessages[I].Role = 'user' then
      Result[I] := TChatMessagePayload.User(AMessages[I].Content)
    else
      Result[I] := TChatMessagePayload.Assistant(AMessages[I].Content);
  end;
end;

constructor TAnthropicAi.Create;
begin
  inherited;
  FAnthropicAI := Anthropic.TAnthropic.Create();
  FAnthropicAI.Token := GetEnvironmentVariable('ANTHROPIC_API_KEY');

  FModelId := LLM_MODEL_ID;
end;

destructor TAnthropicAi.Destroy;
begin

  inherited;
end;

function TAnthropicAi.GetApiKey: string;
begin
  Result := FAnthropicAI.Token;
end;

function TAnthropicAi.GetModel: string;
begin
  Result := LLM_MODEL_ID;
end;

function TAnthropicAi.GetModels: TArray<string>;
var
  LModels: TModels;
  I: Integer;
begin
  LModels := FAnthropicAI.Models.List;
  try
    SetLength(Result, Length(LModels.Data));
    for I := 0 to Length(LModels.Data) - 1 do
      Result[I] := LModels.Data[I].Id;
  finally
    LModels.Free;
  end;
end;

function TAnthropicAi.GetProvider: string;
begin
  Result := 'Anthropic';
end;

function TAnthropicAi.GetProxySettings: TProxySettings;
begin
end;

function TAnthropicAi.GetSummary(const AContent: string): string;
begin
  Result := TAIEngine.GetSummary(Self, AContent);
end;

procedure TAnthropicAi.SetApiKey(const AValue: string);
begin
  FAnthropicAI.Token := AValue;
end;

procedure TAnthropicAi.SetProxySettings(const AProxySettings: TProxySettings);
begin
end;

initialization

ServiceLocator.RegisterClass(TAnthropicAi);

end.
